/*
 * LikelihoodEstimators.cpp
 *
 *  Created on: Nov 19, 2013
 *      Author: nino
 */


#include <LikelihoodEstimators.h>


double calculate_realization_likelihood( statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	double tmpLikelihood;

	if ((epidemicParameters_exp->estimate_T_from_temporal_contacts == 1) && (statistic_parameters->contagion_process_type == LAZY_RECOVERY_SIR_TEMPORAL_NET_FROM_TO)){
		// sample activity times for source node from contact data CAUTION : they are cleared inside of this function
		find_node_activity_times(p_infected_parameters, &(epidemicParameters_exp->T_activity_times), epidemicParameters_exp);
	}

	if (( statistic_parameters->use_mask == 1 ) && (likelihood_fun->likelihood_method != LIKELIHOOD_METHOD_SOFT_MARGIN)){
		printf("realization_similarity_phi_mask() not implemented for this likelihood_method in: calculate_realization_likelihood()  \n"); fflush(stdout);
		exit(-1);
	}

    if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_AVGTOPK){
    	tmpLikelihood = calculate_realization_likelihood_avgTopK( p_infected_parameters, statistic_parameters, epidemicParameters_exp, likelihood_fun);
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_UNBIASED_PRUNNING ){
    	tmpLikelihood = calculate_realization_likelihood_unbiased_prunning( epidemicParameters_exp->p, epidemicParameters_exp->q, p_infected_parameters, statistic_parameters, statistic_parameters->observed_realization, epidemicParameters_exp->T );
    }else if (likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_NAIVE ){
    	tmpLikelihood = calculate_realization_likelihood_naive( statistic_parameters, p_infected_parameters, epidemicParameters_exp, likelihood_fun);
    }else if (likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_SOFT_MARGIN){
    	tmpLikelihood = calculate_realization_likelihood_SOFT_MARGIN( statistic_parameters, p_infected_parameters, epidemicParameters_exp, likelihood_fun);
    }else if (likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_JORDAN ){
    	tmpLikelihood = calculate_realization_likelihood_JORDAN( p_infected_parameters, statistic_parameters->observed_realization );
    }else if (likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_DISTANCE){
    	tmpLikelihood = calculate_realization_likelihood_DISTANCE(p_infected_parameters, statistic_parameters->observed_realization );
    }else if (likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_DMP ){
    	tmpLikelihood = calculate_realization_likelihood_DMP(epidemicParameters_exp->p, epidemicParameters_exp->q, p_infected_parameters, statistic_parameters->observed_realization, epidemicParameters_exp->T);
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_AUCDF ){
    	tmpLikelihood = calculate_realization_likelihood_AUCDF(statistic_parameters, p_infected_parameters, epidemicParameters_exp, likelihood_fun);
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_SOFT_MARGIN_PRUNING){
    	tmpLikelihood =  calculate_realization_likelihood_SOFT_MARGIN_MC_PRUNING(statistic_parameters, p_infected_parameters, epidemicParameters_exp, likelihood_fun);
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_SOFT_MARGIN_A_RANGE){
    	tmpLikelihood =  calculate_realization_likelihood_SOFT_MARGIN_a_range(statistic_parameters, p_infected_parameters, epidemicParameters_exp, likelihood_fun);
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_RAND_UNIFORM ){
    	tmpLikelihood = my_random();
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_CONST_ALL ){
    	tmpLikelihood = 1.0;
    }else if ( likelihood_fun->likelihood_method == LIKELIHOOD_METHOD_UNBIASED ){
    	tmpLikelihood = calculate_realization_likelihood_unbiased( epidemicParameters_exp->p, epidemicParameters_exp->q, p_infected_parameters, statistic_parameters, statistic_parameters->observed_realization, epidemicParameters_exp->T );
    }else{
    	printf("Wrong likelihood_method or not implemendted yet !!! \n");
    	exit(-1);
    }

	return tmpLikelihood;
}

void find_node_activity_times(infected_structure * p_infected_parameters, vector<int> * node_activity_times, contagion_param_struct * epidemicParameters_exp){

	node_activity_times->clear();

	int node_target = p_infected_parameters->start_node;

	for(int i = 0; i < p_infected_parameters->contact_array_1->size(); ++i ){
		int tmp_T = p_infected_parameters->time_contact_array->at(i);
		if (tmp_T <= epidemicParameters_exp->T_end){
			if ((p_infected_parameters->contact_array_1->at(i) == node_target) || (p_infected_parameters->contact_array_2->at(i) == node_target)){  // node is active
				if ((tmp_T >= epidemicParameters_exp->prior_T_low_bound) && (tmp_T <= epidemicParameters_exp->prior_T_high_bound )){ //prior knowledge
					node_activity_times->push_back( tmp_T ); //report time activity
				}
			}else{
				continue; //node is not active at this step
			}
		}else{
			break;
		}
	}

	if (node_activity_times->size() == 0){
		//printf("Worker: No nodes activities for node %d were found after %d in find_node_activity_times() and before %d \n", node_target, epidemicParameters_exp->prior_T_low_bound, epidemicParameters_exp->prior_T_high_bound);
		node_activity_times->push_back( epidemicParameters_exp->prior_T_low_bound );
		/*
		printf("ALL acitivites added ! \n", node_target);
		for(int i = 0; i < p_infected_parameters->contact_array_1->size(); ++i ){
			if (p_infected_parameters->time_contact_array->at(i) < epidemicParameters_exp->T_end){
				if ((p_infected_parameters->contact_array_1->at(i) == node_target) || (p_infected_parameters->contact_array_2->at(i) == node_target)){  // node is active
						node_activity_times->push_back( p_infected_parameters->time_contact_array->at(i) ); //report time activity
						//printf(" %d ", p_infected_parameters->time_contact_array->at(i));
				}else{
					continue; //node is not active at this step
				}
			}else{
				break;
			}
		}
		*/
		return;
	}
}

double calculate_realization_likelihood_avgTopK(infected_structure * p_infected_parameters, statistics * statistic_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	igraph_vector_t * scores_array = (igraph_vector_t *) malloc( sizeof(igraph_vector_t) ); //***HEAP-MEMORY SAFE
	igraph_vector_init( scores_array, statistic_parameters->no_of_iteration_per_cycle );
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

    for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

		run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed);

		VECTOR(*scores_array)[i] = realization_similarity_phi( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
    }

    double tail_ratio = likelihood_fun->likelihood_param.first;
    int k = round( tail_ratio * (double) statistic_parameters->no_of_iteration_per_cycle );
    double avgTopk_stat = average_top_k_scores( scores_array, k );

    igraph_vector_destroy(scores_array);
    free(scores_array);
    //printf("%lf ", avgTopk_stat);
	return avgTopk_stat;
}

double average_top_k_scores( igraph_vector_t * scores_array, int k ){
	igraph_vector_sort( scores_array ) ;
	// ascending

	double tmpScoreSum = 0.0;
	int size = igraph_vector_size( scores_array );
	int position = size - 1;
	for( int i = 0; i < k; ++i){
		tmpScoreSum += (double) VECTOR(*scores_array) [position-i];
	}

	double avgScore = ((double) tmpScoreSum) / k;
	return avgScore;
}

double calculate_realization_likelihood_unbiased_prunning(double p, double q, infected_structure * p_infected_parameters, statistics * statistic_parameters, igraph_vector_t * observed_realization, int  T_value){

	int num_same_realizations = 0;
	int is_prunned;

    for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

    	// ubaceno unutar contagion modela: clear: statistic_parameters->recovered_nodes, dq_infected_nodes, q_infected_nodes_all

    	is_prunned = 0;
		igraph_epidemic_classic_temporal_prunning_identitiy(p_infected_parameters, p, q, &statistic_parameters->recovered_nodes, T_value, observed_realization, &is_prunned);

		if (is_prunned == 1){
			continue;
		}else{
			num_same_realizations += realization_identity( observed_realization, &statistic_parameters->recovered_nodes);
		}

    }

    double likelihood = ((double) num_same_realizations) / ((double) statistic_parameters->no_of_iteration_per_cycle);
	return likelihood;
}

double calculate_realization_likelihood_unbiased(double p, double q, infected_structure * p_infected_parameters, statistics * statistic_parameters, igraph_vector_t * observed_realization, int  T_value){

	int num_same_realizations = 0;

    for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {
    	// ubaceno unutar contagion modela: clear: statistic_parameters->recovered_nodes, dq_infected_nodes, q_infected_nodes_all

		igraph_epidemic_classic_temporal(p_infected_parameters, p, q, &statistic_parameters->recovered_nodes, T_value);

		num_same_realizations += realization_identity( observed_realization, &statistic_parameters->recovered_nodes);
    }

    double likelihood = ((double) num_same_realizations) / ((double) statistic_parameters->no_of_iteration_per_cycle);
	return likelihood;
}

double calculate_realization_likelihood_naive( statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun ){

	igraph_vector_t node_inf_probability; //**** HEAP MEMORY SAFE
	igraph_vector_init( &node_inf_probability , p_infected_parameters->no_of_nodes);
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

    for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

    	run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed);
		igraph_vector_add( &node_inf_probability, &statistic_parameters->recovered_nodes);
    }

    double epsilon;
    if ( (likelihood_fun->likelihood_param.first >= 0.0) && (likelihood_fun->likelihood_param.first <= 1.0) ){
    	epsilon = likelihood_fun->likelihood_param.first;
    }else{
    	epsilon = 0.0000001; //default
    }

    // adding epsilon frequency - smoothing
	igraph_vector_add_constant( &node_inf_probability, (igraph_real_t) epsilon);
	igraph_vector_scale( &node_inf_probability, (double)1. / ((double) statistic_parameters->no_of_iteration_per_cycle + epsilon) );
	// p_i = ( m + e ) / ( n + e )

	double logLikelihood = 0.0;
	for( int i = 0 ; i < igraph_vector_size(statistic_parameters->observed_realization); ++i){
		if ( VECTOR(*statistic_parameters->observed_realization)[i] == 1 ){
			// infected node
			logLikelihood += log( VECTOR(node_inf_probability)[i] );
		}else{
			//not infected
			logLikelihood += log( (1 - (VECTOR(node_inf_probability)[i] ) ) );
		}
	}

	igraph_vector_destroy(&node_inf_probability);
	double likelihood = exp(logLikelihood);
	return likelihood;
}

double calculate_realization_likelihood_SOFT_MARGIN(statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	igraph_vector_t * scores_array = (igraph_vector_t *) malloc( sizeof(igraph_vector_t) ); //***HEAP-MEMORY SAFE
	igraph_vector_init( scores_array, statistic_parameters->no_of_iteration_per_cycle );
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

	for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

		run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed, i);
		if (statistic_parameters->use_mask == 1){
			VECTOR(*scores_array)[i] = realization_similarity_phi_mask( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, statistic_parameters->mask, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
		}else{
			VECTOR(*scores_array)[i] = realization_similarity_phi( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
		}
	}

	double sigma;
	if ( likelihood_fun->likelihood_param.first >= 0.0){
		sigma = likelihood_fun->likelihood_param.first;
	}else{ //default
		sigma = 0.05;
	}

	double likelihood = weighted_soft_margin( scores_array, sigma );

	igraph_vector_destroy(scores_array);
	free(scores_array);
	return likelihood;

}

double calculate_realization_likelihood_SOFT_MARGIN_a_range(statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	igraph_vector_t * scores_array = (igraph_vector_t *) malloc( sizeof(igraph_vector_t) ); //***HEAP-MEMORY SAFE
	igraph_vector_init( scores_array, statistic_parameters->no_of_iteration_per_cycle );
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

	for(int i = 0; i < likelihood_fun->a_array.size(); ++i){//clear
		likelihood_fun->likelihood_theta_a_array.at(i) = 0.0;
	}

	for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

		run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed, i);
		VECTOR(*scores_array)[i] = realization_similarity_phi( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
	}

	double sigma;
	if (likelihood_fun->a_array.size()==0){
		printf("ERROR: The likelihood_fun->a_array.size() is equal to 0 in calculate_realization_likelihood_SOFT_MARGIN_a_range() function !\n"); fflush(stdout);
		exit(-1);
	}

	double likelihood = 0.0;
	for(int i = 0; i < likelihood_fun->a_array.size(); ++i){
		likelihood = weighted_soft_margin( scores_array, likelihood_fun->a_array.at(i) );
		likelihood_fun->likelihood_theta_a_array.at(i) = likelihood;
		//printf("KITA %lf ", likelihood_fun->likelihood_theta_a_array.at(i));
	}


	igraph_vector_destroy(scores_array);
	free(scores_array);
	return 0.0;

}

double calculate_realization_likelihood_SOFT_MARGIN_MC_PRUNING(statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	igraph_vector_t * scores_array = (igraph_vector_t *) malloc( sizeof(igraph_vector_t) ); //***HEAP-MEMORY SAFE
	igraph_vector_init( scores_array, statistic_parameters->no_of_iteration_per_cycle );
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

	//calculate error term
	double phi_boundary;
	if ( likelihood_fun->likelihood_param.second >= 0.0){
		phi_boundary = likelihood_fun->likelihood_param.second;
	}else{
		printf("MC prunning phi_boundary not defined !!! \n");
		exit(-1);
	}

	double error = 1.0 - phi_boundary;
	int error_nodes = (int) ceil(p_infected_parameters->no_of_nodes * error);
	int is_pruned;

	for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

		// ubaceno unutar contagion modela *****
		//clear_vector_idx(p_infected_parameters->q_infected_nodes_all, &statistic_parameters->recovered_nodes);
		//igraph_dqueue_clear(p_infected_parameters->dq_infected_nodes);
		//igraph_dqueue_clear(p_infected_parameters->q_infected_nodes_all);

		is_pruned = 0;
		if (statistic_parameters->contagion_process_type == NAIVE_SIR_UNWEIGHTED_NET){
			igraph_epidemic_classic_MC_pruning(p_infected_parameters, epidemicParameters_exp->p, epidemicParameters_exp->q, &statistic_parameters->recovered_nodes, epidemicParameters_exp->T, statistic_parameters->observed_realization, &is_pruned, error_nodes);
		}else{
			printf("MC prunning not implemented for other processes !!! \n");
			exit(-1);
		}

		if (is_pruned == 1){
			VECTOR(*scores_array)[i] = 0.0;
		}else{
			VECTOR(*scores_array)[i] = realization_similarity_phi( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
		}

	}

	double sigma;
	if ( likelihood_fun->likelihood_param.first >= 0.0){
		sigma = likelihood_fun->likelihood_param.first;
	}else{ //default
		sigma = 0.05;
	}

	double likelihood = weighted_soft_margin( scores_array, sigma );

	igraph_vector_destroy(scores_array);
	free(scores_array);
	return likelihood;

}

double weighted_soft_margin( igraph_vector_t * scores_array, double sigma ){

	double tmpScoreSum = 0.0;
	int size = igraph_vector_size( scores_array );

	for( int i = 0; i < size; ++i){
		double tmp_sim = (double) VECTOR(*scores_array) [i];
		tmpScoreSum += general_dirac( tmp_sim - 1.0, sigma );
	}

	double avgScore = ((double) tmpScoreSum) / (double) size;
	if ((avgScore < 0.0) || (avgScore > 1.0)){
		printf(" ERROR in weighted_soft_margin fun (), likelihood %lf out of bounds, sigma %lf \n", avgScore, sigma );
		exit(-1);
	}
	//printf("OK: %lf ", avgScore);
	return avgScore;
}

double general_dirac( double x, double sigma){

	//double norm = (double) 1.0 / general_dirac_norm(sigma);
	double score = 1.0 *  exp(-1.0*(x*x) / (sigma*sigma) );
	return score;
}

double general_dirac_norm( double sigma ){
	/* Integrate[Exp[-((x - 1)^2/a^2)], {x, 0, 1}]
	 * 1/2 a Sqrt[\[Pi]] Erf[1/a]
	 * http://www.cplusplus.com/reference/cmath/erf/
	 */

	double tmp = (double) 1.0 / sigma;
	double norm = 0.5 * sigma * (double) SQRTPI * erf(tmp);

	return norm;
}

double calculate_realization_likelihood_AUCDF(statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp, likelihood_struct * likelihood_fun){

	igraph_vector_t * scores_array = (igraph_vector_t *) malloc( sizeof(igraph_vector_t) ); //***HEAP-MEMORY SAFE
	igraph_vector_init( scores_array, statistic_parameters->no_of_iteration_per_cycle );
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();

    for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

    	run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed);
		VECTOR(*scores_array)[i] = realization_similarity_phi( statistic_parameters->observed_realization, &statistic_parameters->recovered_nodes, likelihood_fun->phi_type , p_infected_parameters->no_of_nodes);
    }

    double AUCDF_likelihood = 1.0 - areaUnderCDF(scores_array);

    igraph_vector_destroy(scores_array);
    free(scores_array);

	return AUCDF_likelihood;
}

double areaUnderCDF(igraph_vector_t * scores_array){
	igraph_vector_sort( scores_array ) ;
	// ascending

	int size = igraph_vector_size( scores_array );
	double CDF_jump = ((double) 1.0 ) / ((double)size);
	double height = 0.0;
	double AUCDF = 0.0;
	for( int i = 0; i < size-1; ++i){
		double delta = ( VECTOR(*scores_array)[i+1] ) - ( VECTOR(*scores_array)[i] );
		height += CDF_jump;
		AUCDF += ( delta * height);
	}
	double delta = 1.0 - ( VECTOR(*scores_array)[size-1] );
	height += CDF_jump;
	AUCDF += ( delta * height);

	return AUCDF;
}

double calculate_realization_likelihood_JORDAN(infected_structure * p_infected_parameters, igraph_vector_t * infected_nodes){

	igraph_t subgraph_inf;
	igraph_empty(&subgraph_inf, (igraph_integer_t) p_infected_parameters->no_of_nodes, IGRAPH_UNDIRECTED);
	find_infected_subgraph(p_infected_parameters, infected_nodes, &subgraph_inf);

	igraph_matrix_t geodesic_paths;
	igraph_matrix_init( &geodesic_paths, (long int) 1, (long int) p_infected_parameters->no_of_nodes);
	#ifdef IGRAPH_VERSION_0_6_5
	igraph_shortest_paths( &subgraph_inf, &geodesic_paths, igraph_vss_1((igraph_integer_t) p_infected_parameters->start_node), igraph_vss_all(), IGRAPH_ALL);
	#else
	igraph_shortest_paths( &subgraph_inf, &geodesic_paths, igraph_vss_1((igraph_integer_t) p_infected_parameters->start_node), IGRAPH_ALL);
	#endif


	double jordan_weight = 0.0;
	for ( long int node_tmp; node_tmp < p_infected_parameters->no_of_nodes; ++node_tmp ){
		if (VECTOR(*infected_nodes)[node_tmp] == 1){
			double tmp_dist = (double) MATRIX( geodesic_paths, 0 ,(int) node_tmp );
			if ( tmp_dist > jordan_weight ){
				jordan_weight = tmp_dist;
			}
		}
	}

	igraph_matrix_destroy(&geodesic_paths);
	igraph_destroy(&subgraph_inf);

	double likelihood = (double) 1.0 / jordan_weight;

	return likelihood;
}

void find_infected_subgraph(infected_structure * p_infected_parameters, igraph_vector_t * infected_nodes, igraph_t * inf_subgraph){
	//this procedure is slow !!!

	int init_size = (int) igraph_vector_sum ( infected_nodes );
	igraph_vector_t edges_infected;
	igraph_vector_init ( &edges_infected, 0 );
	igraph_vector_reserve(&edges_infected, 2*init_size);

	//printf("Infected subgraph \n");
	// undirected network
	for ( long int node_tmp; node_tmp < p_infected_parameters->no_of_nodes; ++node_tmp){
		if  ( VECTOR(*infected_nodes)[node_tmp] == 0 ){
			continue;
			// if the node is not infected continue to next node
		}else{
			igraph_vector_t *neis = igraph_adjlist_get(p_infected_parameters->al, node_tmp);

			for (int i = 0; i < (int) igraph_vector_size(neis); ++i) {
				long int current_neigh = (long int) VECTOR(*neis)[i];

				if ( VECTOR(*infected_nodes)[current_neigh] == 1 ){
					// if both neighbouring nodes are infected we add the edge to new graph

					igraph_vector_push_back (&edges_infected, (igraph_integer_t) node_tmp);
					igraph_vector_push_back (&edges_infected, (igraph_integer_t) current_neigh);
					//igraph_add_edge(inf_subgraph, (igraph_integer_t) node_tmp, (igraph_integer_t) current_neigh);
					//printf(" %d -> %d ", (int) node_tmp, (int) current_neigh);
				}
			}
		}
	}

	//printf("vector length: %d \n", (int) igraph_vector_size(&edges_infected) );
	// add edges in stream
	igraph_add_edges(inf_subgraph, &edges_infected, 0);

	igraph_vector_destroy( &edges_infected );
	//eliminate mulitedges
	#ifdef IGRAPH_VERSION_0_6_5
	igraph_simplify(inf_subgraph, (igraph_bool_t) 1, (igraph_bool_t) 1,0);
	#else
	igraph_simplify(inf_subgraph, (igraph_bool_t) 1, (igraph_bool_t) 1);
	#endif
}

void find_infected_subgraph_temporal(infected_structure * p_infected_parameters, igraph_vector_t * infected_nodes, igraph_t * inf_subgraph){
	//this procedure is slow !!!

	int init_size = (int) igraph_vector_sum ( infected_nodes );
	igraph_vector_t edges_infected;
	igraph_vector_init ( &edges_infected, 0 );
	igraph_vector_reserve(&edges_infected, 2*init_size);

	//printf("Infected subgraph \n");
	// undirected network
	for ( long int node_tmp; node_tmp < p_infected_parameters->no_of_nodes; ++node_tmp){
		if  ( VECTOR(*infected_nodes)[node_tmp] == 0 ){
			continue;
			// if the node is not infected continue to next node
		}else{
			igraph_vector_t neis;
			igraph_vector_init ( &neis, p_infected_parameters->no_of_nodes);
			igraph_neighbors(p_infected_parameters->p_graph, &neis, (igraph_integer_t) node_tmp, IGRAPH_ALL);

			for (int i = 0; i < (int) igraph_vector_size(&neis); ++i) {
				long int current_neigh = (long int) VECTOR(neis)[i];

				if ( VECTOR(*infected_nodes)[current_neigh] == 1 ){
					// if both neighbouring nodes are infected we add the edge to new graph

					igraph_vector_push_back (&edges_infected, (igraph_integer_t) node_tmp);
					igraph_vector_push_back (&edges_infected, (igraph_integer_t) current_neigh);
					//igraph_add_edge(inf_subgraph, (igraph_integer_t) node_tmp, (igraph_integer_t) current_neigh);
					//printf(" %d -> %d ", (int) node_tmp, (int) current_neigh);
				}
			}
		}
	}

	//printf("vector length: %d \n", (int) igraph_vector_size(&edges_infected) );
	// add edges in stream
	igraph_add_edges(inf_subgraph, &edges_infected, 0);

	igraph_vector_destroy( &edges_infected );
	//eliminate mulitedges
	#ifdef IGRAPH_VERSION_0_6_5
	igraph_simplify(inf_subgraph, (igraph_bool_t) 1, (igraph_bool_t) 1,0);
	#else
	igraph_simplify(inf_subgraph, (igraph_bool_t) 1, (igraph_bool_t) 1);
	#endif
}

double calculate_realization_likelihood_DISTANCE(infected_structure * p_infected_parameters, igraph_vector_t * infected_nodes){

	igraph_t subgraph_inf;
	igraph_empty(&subgraph_inf, (igraph_integer_t) p_infected_parameters->no_of_nodes, IGRAPH_UNDIRECTED);
	find_infected_subgraph(p_infected_parameters, infected_nodes, &subgraph_inf);

	igraph_matrix_t geodesic_paths;
	igraph_matrix_init( &geodesic_paths, (long int) 1, (long int) p_infected_parameters->no_of_nodes);

	#ifdef IGRAPH_VERSION_0_6_5
	igraph_shortest_paths( &subgraph_inf, &geodesic_paths, igraph_vss_1((igraph_integer_t) p_infected_parameters->start_node), igraph_vss_all(), IGRAPH_ALL);
	#else
	igraph_shortest_paths( &subgraph_inf, &geodesic_paths, igraph_vss_1((igraph_integer_t) p_infected_parameters->start_node), IGRAPH_ALL);
	#endif

	double distance_weight = 0.0;
	for ( long int node_tmp; node_tmp < p_infected_parameters->no_of_nodes; ++node_tmp ){
		if (VECTOR(*infected_nodes)[node_tmp] == 1){
			double tmp_dist = (double) MATRIX( geodesic_paths, 0 ,(int) node_tmp );
			distance_weight += tmp_dist;
		}
	}

	igraph_matrix_destroy(&geodesic_paths);
	igraph_destroy(&subgraph_inf);
	double likelihood = (double) 1.0 / distance_weight;

	return likelihood;
}

double calculate_realization_likelihood_DMP(double p, double q, infected_structure * p_infected_parameters, igraph_vector_t * observed_realization, int T_value){

	igraph_vector_t node_inf_probability;
	igraph_vector_init( &node_inf_probability , p_infected_parameters->no_of_nodes);

	DMP_algorithm(p, q, p_infected_parameters, T_value, &node_inf_probability);

	double logLikelihood = 0.0;
	for( int i = 0 ; i < igraph_vector_size(observed_realization); ++i){
		if ( VECTOR(*observed_realization)[i] == 1 ){
			// infected node
			logLikelihood += log( VECTOR(node_inf_probability)[i] );
		}else{
			//not infected
			logLikelihood += log( (1 - (VECTOR(node_inf_probability)[i] ) ) );
		}
	}


	double likelihood = exp(logLikelihood);
	return likelihood;

}

void copy_likelihood_vector(double * source, double * dest, int size){
	for(int i = 0; i< size; ++i){
		dest[i] = source[i];
	}
}

int source_prob_distr_converged(double * likeihood_new, double * likelihood_old, int size, double convergence_threshold){
	// max{prob_1(x) - prob_2(x)} < convergence_threshold -> converged !!!

	double norm_sum_new = 0.0;
	double norm_sum_old = 0.0;
	for(int i=0; i < size; ++i){
		norm_sum_new += likeihood_new[i];
		norm_sum_old += likelihood_old[i];
	}

	if (norm_sum_new == 0.0){
		return 0;
	}

	if(norm_sum_old == 0.0){
		return 0;
	}

	for(int i = 0; i< size; ++i){
		double prob_new = likeihood_new[i] / norm_sum_new;
		double prob_old = likelihood_old[i] / norm_sum_old;
		if ( abs(prob_new - prob_old) > convergence_threshold ){
			return 0; // did not converged
		}
	}

	return 1; //converged

}

int recalc_likelihoods_old_new(double * likelihood_new, double * likelihood_old, int size, int num_sim_new, int num_sim_old){
	// append new likelihood results to old ones and renormalize them


	for(int i=0; i < size; ++i){
		likelihood_new[i] *= num_sim_new; // denormalize likelihood
		likelihood_old[i] *= num_sim_old; // denormalize likelihood

		likelihood_new[i] += likelihood_old[i]; // add old contribution to new contribution
		likelihood_new[i] /= (num_sim_old + num_sim_new); // re-normalize likelihood
	}

}

int rank_ML_converged(double * likelihood_new, double * likelihood_old, int size){
	// return 1 id same node is ranked on position 1 in boths arrays

	double norm_sum_new = 0.0; double norm_sum_old = 0.0;
	int ML_idx = -1; int ML_idx_old = -1;
	double ML_value = -1.0; double ML_value_old = -1.0;

	//find ML node in both arrays
	for(int i=0; i < size; ++i){
		norm_sum_new += likelihood_new[i];
		if (likelihood_new[i] > ML_value){
			ML_idx = i;
			ML_value = likelihood_new[i];
		}

		norm_sum_old += likelihood_old[i];
		if (likelihood_old[i] > ML_value_old){
			ML_idx_old = i;
			ML_value_old = likelihood_old[i];
		}
	}

	if ((norm_sum_new == 0.0)||(norm_sum_old == 0.0)){
		return 0;
	}

	if (ML_idx_old != ML_idx){
		return 0;
	}else{
		return 1;
	}


}

int source_prob_distr_ML_converged(double * likeihood_new, double * likelihood_old, int size, double convergence_threshold){
	// (prob_ML(x1) - prob_ML(x2))/prob_ML(x1) < convergence_threshold : relative change of ML node
	// Ideally we want to find the ML node and its proability

	double norm_sum_new = 0.0;
	double norm_sum_old = 0.0;
	int ML_idx = -1;
	double ML_value = -1.0;
	for(int i=0; i < size; ++i){
		norm_sum_new += likeihood_new[i];
		norm_sum_old += likelihood_old[i];
		if (likeihood_new[i] > ML_value){
			ML_idx = i;
			ML_value = likeihood_new[i];
		}
	}

	if (norm_sum_new == 0.0){
		return 0;
	}

	if(norm_sum_old == 0.0){
		return 0;
	}

	ML_value = ML_value / norm_sum_new;
	double prob_old_ML = likelihood_old[ML_idx] / norm_sum_old;
	double rel_ML_error = abs(prob_old_ML - ML_value) / ML_value;
	if ( rel_ML_error > convergence_threshold ){
		return 0; // did not converged
	}

	/*
	for(int i = 0; i< size; ++i){
		double prob_new = likeihood_new[i] / norm_sum_new;
		double prob_old = likelihood_old[i] / norm_sum_old;
		double rel_change = abs(prob_new - prob_old) / ML_value ;
		if ( rel_change > convergence_threshold ){
			return 0; // did not converged
		}
	}
	*/
	return 1; //converged

}

double source_prob_distr_ML_rel_error(double * likeihood_new, double * likelihood_old, int size){
	// returns (prob_ML(x1) - prob_ML(x2))/prob_ML(x1): relative change of ML node

	double norm_sum_new = 0.0;
	double norm_sum_old = 0.0;
	int ML_idx = -1;
	double ML_value = -1.0;
	for(int i=0; i < size; ++i){
		norm_sum_new += likeihood_new[i];
		norm_sum_old += likelihood_old[i];
		if (likeihood_new[i] > ML_value){ //find ML node
			ML_idx = i;
			ML_value = likeihood_new[i];
		}
	}

	if ((norm_sum_new == 0.0) || (norm_sum_old == 0.0)){
		return 100000; //big error inf
	}


	ML_value = ML_value / norm_sum_new;
	double prob_old_ML = likelihood_old[ML_idx] / norm_sum_old;
	double rel_ML_error = abs(prob_old_ML - ML_value) / ML_value;

	return rel_ML_error;

}

void generate_realizations_for_export(statistics * statistic_parameters, infected_structure * p_infected_parameters, contagion_param_struct * epidemicParameters_exp){

	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count() * getpid();
	FILE *fp = fopen("MonteCarloRealizations.txt","w");
	fprintf(fp, "#Node: %d, p: %lf, q: %lf, T: %d \n", p_infected_parameters->start_node, epidemicParameters_exp->p, epidemicParameters_exp->q, epidemicParameters_exp->T);

	for (long int i = 0; i < statistic_parameters->no_of_iteration_per_cycle; ++i) {

		run_contagion_model(p_infected_parameters, statistic_parameters->contagion_process_type, &statistic_parameters->recovered_nodes, epidemicParameters_exp, seed);
		for(int i=0; i < igraph_vector_size(&statistic_parameters->recovered_nodes); ++i){
			if ( VECTOR(statistic_parameters->recovered_nodes)[i] == 1 ){
				fprintf(fp, "%d ", i);
			}
		}
		fprintf(fp, " \n");
	}
	fclose(fp);
}
